using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Data;
using System.Windows.Forms;

namespace EQTrack
{
	/// <summary>
	/// Summary description for GradientPanel.
	/// </summary>
	public class GradientPanel : System.Windows.Forms.Panel
	{
		private bool mAutoSize = false;
		private bool mShaded = false;
		private Color mStartColor = Color.Gray;
		private Color mEndColor = Color.White;
		private LinearGradientMode mGradientMode = LinearGradientMode.Vertical;

		public event EventHandler OnInitialise = null;
		public event EventHandler OnFinish = null;

		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		public GradientPanel()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			//this.SetStyle(ControlStyles.SupportsTransparentBackColor, true);
			this.SetStyle(ControlStyles.ContainerControl, true);
			this.SetStyle(ControlStyles.ResizeRedraw, true);
			this.SetStyle(ControlStyles.UserPaint, true);
			this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			this.SetStyle(ControlStyles.DoubleBuffer, true);
		}

		public void Initialise(object sender, EventArgs e)
		{
			if (OnInitialise != null)
				OnInitialise(sender, e);
		}

		public void Finish(object sender, EventArgs e)
		{
			if (OnFinish != null)
				OnFinish(sender, e);
		}

		[DefaultValue(false)]
		public override bool AutoSize {
			get {
				return mAutoSize;
			}
			set {
				if (mAutoSize != value) {
					mAutoSize = value;
					AutoResize();
				}
			}
		}

		[DefaultValue(false)]
		public bool Shaded {
			get {
				return mShaded;
			}
			set {
				if (mShaded != value) {
					mShaded = value;
					this.Refresh();
				}
			}
		}

		[DefaultValue(LinearGradientMode.Vertical)]
		public LinearGradientMode GradientMode {
			get {
				return mGradientMode;
			}
			set {
				if (mGradientMode != value) {
					mGradientMode = value;
					this.Refresh();
				}
			}
		}

		public Color StartColor {
			get {
				return mStartColor;
			}
			set {
				if (mStartColor != value) {
					mStartColor = value;
					this.Refresh();
				}
			}
		}

		public Color EndColor {
			get {
				return mEndColor;
			}
			set {
				if (mEndColor != value) {
					mEndColor = value;
					this.Refresh();
				}
			}
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		protected override void OnPaint(PaintEventArgs e) {
			if (mShaded && this.Width > 0) {
				if (mGradientMode == LinearGradientMode.Vertical && this.Height == 2) {
					e.Graphics.DrawLine(new Pen(mStartColor), 0, 0, this.Width, 0);
					e.Graphics.DrawLine(new Pen(mEndColor), 0, 1, this.Width, 1);
				} else if (this.Height > 0)
					e.Graphics.FillRectangle(new LinearGradientBrush(new Rectangle(0, 0, this.Width, this.Height), mStartColor,
						mEndColor, mGradientMode), 0, 0, this.Width, this.Height);
			}
			base.OnPaint(e);
		}

		private void AutoResize() {
			if (mAutoSize) {
				int MaxHeight = 0;
				for (int i=0; i<this.Controls.Count; i++) {
					if (this.Controls[i].Visible && this.Controls[i].Top + this.Controls[i].Height > MaxHeight && this.Controls[i].Height > 0)
						MaxHeight = this.Controls[i].Top + this.Controls[i].Height;
				}
				if (MaxHeight == 0)
					this.Height = 0;
				else
					this.Height = MaxHeight + this.DockPadding.Bottom;
			}
		}

		protected override void OnControlAdded(ControlEventArgs e) {
			e.Control.Resize += new EventHandler(Control_Resize);
			e.Control.Move += new EventHandler(Control_Resize);
			e.Control.VisibleChanged += new EventHandler(Control_Resize);
			base.OnControlAdded (e);
			AutoResize();
		}

		protected override void OnControlRemoved(ControlEventArgs e) {
			e.Control.Resize -= new EventHandler(Control_Resize);
			e.Control.Move -= new EventHandler(Control_Resize);
			e.Control.VisibleChanged -= new EventHandler(Control_Resize);
			base.OnControlRemoved (e);
			AutoResize();
		}

		private void Control_Resize(object sender, EventArgs e) {
			AutoResize();
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.SuspendLayout();
			// 
			// SetupPanel
			// 
			this.Name = "SetupPanel";
			this.Size = new System.Drawing.Size(400, 125);
			this.ResumeLayout(false);

		}
		#endregion
	}

	/// <summary>
	/// Summary description for CustomPanel.
	/// </summary>
	public class CustomPanel : GradientPanel
	{
		public CustomPanel()
		{
			base.StartColor = Color.White;
			base.EndColor = Color.Gainsboro;
			base.Shaded = true;
		}
	}


	/// <summary>
	/// Summary description for CustomRoundPanel.
	/// </summary>
	public class CustomRoundPanel : GradientPanel
	{
		protected Bitmap ButtonBottom = CustomPanelResources.ButtonBottom;
		protected Bitmap ButtonBottomLeft = CustomPanelResources.ButtonBottomLeft;
		protected Bitmap ButtonBottomRight = CustomPanelResources.ButtonBottomRight;
		protected Bitmap ButtonLeft = CustomPanelResources.ButtonLeft;
		protected Bitmap ButtonRight = CustomPanelResources.ButtonRight;
		protected Bitmap ButtonTop = CustomPanelResources.ButtonTop;
		protected Bitmap ButtonTopLeft = CustomPanelResources.ButtonTopLeft;
		protected Bitmap ButtonTopRight = CustomPanelResources.ButtonTopRight;

		private bool IsMouseHover = false;
		private bool IsMouseDown = false;

		public CustomRoundPanel()
		{
		}

		private Color mPanelColor = SystemColors.Window;
		[
		Category("Appearance"),
		]
		public Color PanelColor
		{
			get
			{
				return mPanelColor;
			}
			set
			{
				if (mPanelColor != value)
				{
					mPanelColor = value;
					this.Refresh();
				}
			}
		}

		private Color mPanelColorHover = SystemColors.Highlight;
		[
		Category("Appearance"),
		]
		public Color PanelColorHover
		{
			get
			{
				return mPanelColorHover;
			}
			set
			{
				if (mPanelColorHover != value)
				{
					mPanelColorHover = value;
					this.Refresh();
				}
			}
		}

		private Color mPanelColorDown = SystemColors.Highlight;
		[
		Category("Appearance"),
		]
		public Color PanelColorDown
		{
			get
			{
				return mPanelColorDown;
			}
			set
			{
				if (mPanelColorDown != value)
				{
					mPanelColorDown = value;
					this.Refresh();
				}
			}
		}

		private int mGradientScale = 40;
		public int GradientScale
		{
			get
			{
				return mGradientScale;
			}
			set
			{
				if (mGradientScale != value)
				{
					mGradientScale = value;
					this.Refresh();
				}
			}
		}

		protected override void OnMouseEnter(EventArgs e)
		{
			base.OnMouseEnter(e);
			IsMouseHover = true;
			if (mPanelColorHover != mPanelColor)
				this.Refresh();
		}

		protected override void OnMouseLeave(EventArgs e)
		{
			base.OnMouseLeave(e);
			IsMouseHover = false;
			if (mPanelColorHover != mPanelColor)
				this.Refresh();
		}

		protected override void OnMouseDown(MouseEventArgs e)
		{
			base.OnMouseDown(e);
			IsMouseDown = true;
			this.Refresh();
		}

		protected override void OnMouseUp(MouseEventArgs e)
		{
			base.OnMouseUp(e);
			IsMouseDown = false;
			this.Refresh();
		}

		protected override void OnPaint(PaintEventArgs e)
		{
			Color ThisPanelColor = mPanelColor;
			if (!this.Enabled)
				ThisPanelColor = SystemColors.ControlDark;
			else if (IsMouseDown)
				ThisPanelColor = mPanelColorDown;
			else if (IsMouseHover)
				ThisPanelColor = mPanelColorHover;
			if (base.Shaded && this.Width > 0 && this.Height > 0)
			{
				int R = ThisPanelColor.R;
				R -= mGradientScale;
				if (R < 0)
					R = 0;
				int G = ThisPanelColor.G;
				G -= mGradientScale;
				if (G < 0)
					G = 0;
				int B = ThisPanelColor.B;
				B -= mGradientScale;
				if (B < 0)
					B = 0;
				Color ThisEndColor = Color.FromArgb(R, G, B);
				e.Graphics.FillRectangle(new LinearGradientBrush(new Rectangle(2, 2, this.Width - 8, this.Height - 8), ThisPanelColor,
						ThisEndColor, base.GradientMode), 2, 2, this.Width - 8, this.Height - 8);
			} else
				e.Graphics.FillRectangle(new SolidBrush(ThisPanelColor), 2, 2, this.Width - 8, this.Height - 8);

			e.Graphics.DrawImage(ButtonTopLeft, new Rectangle(new Point(0, 0), ButtonTopLeft.Size), new Rectangle(new Point(0, 0), ButtonTopLeft.Size), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonTopRight, new Rectangle(new Point(this.Width - ButtonTopRight.Width, 0), ButtonTopRight.Size), new Rectangle(new Point(0, 0), ButtonTopRight.Size), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonBottomRight, new Rectangle(new Point(this.Width - ButtonBottomRight.Width, this.Height - ButtonBottomRight.Height), ButtonBottomRight.Size), new Rectangle(new Point(0, 0), ButtonBottomRight.Size), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonBottomLeft, new Rectangle(new Point(0, this.Height - ButtonBottomLeft.Height), ButtonBottomLeft.Size), new Rectangle(new Point(0, 0), ButtonBottomLeft.Size), GraphicsUnit.Pixel);

			e.Graphics.DrawImage(ButtonTop, new Rectangle(ButtonTopLeft.Width, 0, this.Width - ButtonTopLeft.Width - ButtonTopRight.Width, ButtonTop.Height), new Rectangle(0, 0, ButtonTop.Width - 1, ButtonTop.Height), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonLeft, new Rectangle(0, ButtonTopLeft.Height, ButtonLeft.Width, this.Height - ButtonTopLeft.Height - ButtonBottomLeft.Height), new Rectangle(0, 0, ButtonLeft.Width, ButtonLeft.Height - 1), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonBottom, new Rectangle(ButtonBottomLeft.Width, this.Height - ButtonBottom.Height, this.Width - ButtonBottomLeft.Width - ButtonBottomRight.Width, ButtonBottom.Height), new Rectangle(0, 0, ButtonBottom.Width - 1, ButtonBottom.Height), GraphicsUnit.Pixel);
			e.Graphics.DrawImage(ButtonRight, new Rectangle(this.Width - ButtonRight.Width, ButtonTopRight.Height, ButtonRight.Width, this.Height - ButtonTopRight.Height - ButtonBottomRight.Height), new Rectangle(0, 0, ButtonRight.Width, ButtonRight.Height - 1), GraphicsUnit.Pixel);
		}
	}
}
